//  $Id: fmfloppy.cc,v 1.2 2001/07/13 15:15:06 nishi Exp $
//
//  Copyright (C) 2001 Shouhei Nishi.
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

extern "C" {
#include <errno.h>
}

#include "bochs.h"
#if BX_EMULATION_TOWNS
#define LOG_THIS bx_floppy.


bx_floppy_ctrl_c bx_floppy;

#if BX_USE_FD_SMF
#define this (&bx_floppy)
#endif



/* for main status register */
#define FD_MS_MRQ  0x80
#define FD_MS_DIO  0x40
#define FD_MS_NDMA 0x20
#define FD_MS_BUSY 0x10
#define FD_MS_ACTD 0x08
#define FD_MS_ACTC 0x04
#define FD_MS_ACTB 0x02
#define FD_MS_ACTA 0x01

#define FROM_FLOPPY 10
#define TO_FLOPPY   11

#define FLOPPY_DMA_CHAN 2



bx_floppy_ctrl_c::bx_floppy_ctrl_c(void)
{
	setprefix("[FDD ]");
	settype(FDLOG);
	BX_DEBUG(("Init.\n"));
}

bx_floppy_ctrl_c::~bx_floppy_ctrl_c(void)
{
	// nothing for now
	BX_DEBUG(("Exit.\n"));
}


  void
bx_floppy_ctrl_c::init(bx_devices_c *d, bx_cmos_c *cmos)
{
  int i;

  BX_FD_THIS devices = d;

  BX_FD_THIS devices->register_irq(6, "Floppy Drive");
  for (unsigned addr=0x0200; addr<=0x020e; addr+=2) {
    BX_FD_THIS devices->register_io_read_handler(this, read_handler,
                                      addr, "Floppy Drive");
    BX_FD_THIS devices->register_io_write_handler(this, write_handler,
                                      addr, "Floppy Drive");
    }


  BX_FD_THIS s.num_supported_floppies = 0;

  //
  // Floppy A setup
  //
  BX_FD_THIS s.media[0].bytes_per_sector  = 0;
  BX_FD_THIS s.media[0].sectors_per_track = 0;
  BX_FD_THIS s.media[0].tracks            = 0;
  BX_FD_THIS s.media[0].heads             = 0;
  BX_FD_THIS s.media[0].sectors           = 0;
  BX_FD_THIS s.media[0].type              = BX_FLOPPY_NONE;
  BX_FD_THIS s.media[0].fd = -1;
  BX_FD_THIS s.media_present[0] = 0;


  if (bx_options.floppya.type != BX_FLOPPY_NONE) {
    BX_FD_THIS s.num_supported_floppies++;
    if ( bx_options.floppya.initial_status == BX_INSERTED) {
      if (evaluate_media(bx_options.floppya.type, bx_options.floppya.path,
                   & BX_FD_THIS s.media[0]))
        BX_FD_THIS s.media_present[0] = 1;
      }
    }


  //
  // Floppy B setup
  //
  BX_FD_THIS s.media[1].bytes_per_sector  = 0;
  BX_FD_THIS s.media[1].sectors_per_track = 0;
  BX_FD_THIS s.media[1].tracks            = 0;
  BX_FD_THIS s.media[1].heads             = 0;
  BX_FD_THIS s.media[1].sectors           = 0;
  BX_FD_THIS s.media[1].type              = BX_FLOPPY_NONE;
  BX_FD_THIS s.media[1].fd = -1;
  BX_FD_THIS s.media_present[1] = 0;

  if (bx_options.floppyb.type != BX_FLOPPY_NONE) {
    BX_FD_THIS s.num_supported_floppies++;
    if ( bx_options.floppyb.initial_status == BX_INSERTED) {
      if (evaluate_media(bx_options.floppyb.type, bx_options.floppyb.path,
                   & BX_FD_THIS s.media[1]))
        BX_FD_THIS s.media_present[1] = 1;
      }
    }

  BX_FD_THIS s.selected_drive=-1;
  for(i=0;i<4;i++) {
    BX_FD_THIS s.INUSE[i]=0;
    BX_FD_THIS s.HISPD[i]=0;
  }
  BX_FD_THIS s.DRVCHG=0;
  BX_FD_THIS s.CLKSEL=0;
  BX_FD_THIS s.MOTOR=0;
  BX_FD_THIS s.HDISEL=0;
  BX_FD_THIS s.DDEN=0;
  BX_FD_THIS s.IRQMSK=0;

  BX_FD_THIS s.floppy_timer_index =
    bx_pc_system.register_timer( this, timer_handler,
      bx_options.floppy_command_delay, 0,0);

  BX_INFO(("bx_options.floppy_command_delay = %u\n",
    (unsigned) bx_options.floppy_command_delay));
}



  void
bx_floppy_ctrl_c::reset(unsigned source)
{
  BX_FD_THIS s.command_complete = 1; /* waiting for new command */

  BX_FD_THIS s.track=0;
  BX_FD_THIS s.sector=0;
  BX_FD_THIS s.data=0;
  BX_FD_THIS s.status=0x04;
  BX_FD_THIS s.floppy_buffer_index = 0;
}


  // static IO port read callback handler
  // redirects to non-static class handler to avoid virtual functions

  Bit32u
bx_floppy_ctrl_c::read_handler(void *this_ptr, Bit32u address, unsigned io_len)
{
#if !BX_USE_FD_SMF
  bx_floppy_ctrl_c *class_ptr = (bx_floppy_ctrl_c *) this_ptr;

  return( class_ptr->read(address, io_len) );
}


  /* reads from the floppy io ports */
  Bit32u
bx_floppy_ctrl_c::read(Bit32u address, unsigned io_len)
{
#else
  UNUSED(this_ptr);
#endif  // !BX_USE_FD_SMF

  if (io_len > 1)
    BX_PANIC(("io read from address %08x, len=%u\n",
             (unsigned) address, (unsigned) io_len));

// ???
//if (bx_cpu.cs.selector.value != 0xf000) {
//  BX_INFO(("BIOS: floppy: read access to port %04x\n", (unsigned) address);
//  }

  if (bx_dbg.floppy)
    BX_INFO(("read access to port %04x\n", (unsigned) address));

  switch (address) {
  case 0x0200:
    return(BX_FD_THIS s.status);

  case 0x0202:
    return(BX_FD_THIS s.track);

  case 0x0208:
    if(BX_FD_THIS s.selected_drive<0 ||
       BX_FD_THIS s.selected_drive>=2 ) {
      return (0x01);
    }
    return (0x07);

  case 0x020e:
    return(BX_FD_THIS s.DRVCHG ? 0x01 : 0x00);
  }

  BX_PANIC(("floppy: io_read: bailing\n"));
  return(0);
}


  // static IO port write callback handler
  // redirects to non-static class handler to avoid virtual functions

  void
bx_floppy_ctrl_c::write_handler(void *this_ptr, Bit32u address, Bit32u value, unsigned io_len)
{
#if !BX_USE_FD_SMF
  bx_floppy_ctrl_c *class_ptr = (bx_floppy_ctrl_c *) this_ptr;

  class_ptr->write(address, value, io_len);
}

  /* writes to the floppy io ports */
  void
bx_floppy_ctrl_c::write(Bit32u address, Bit32u value, unsigned io_len)
{
#else
  UNUSED(this_ptr);
#endif  // !BX_USE_FD_SMF

  if (io_len > 1)
    BX_PANIC(("io write to address %08x, len=%u\n",
             (unsigned) address, (unsigned) io_len));

// ???
//if (bx_cpu.cs.selector.value != 0xf000) {
//  BX_INFO(("BIOS: floppy: write access to port %04x, value=%02x\n",
//      (unsigned) address, (unsigned) value));
//  }

  if (bx_dbg.floppy)
    BX_INFO(("write access to port %04x, value=%02x\n",
      (unsigned) address, (unsigned) value));

  switch (address) {
   case 0x0200:
     if(value == 0xfe) return;
     if(value == 0xd0) return;
     if(BX_FD_THIS s.selected_drive<0 ||
	BX_FD_THIS s.selected_drive>=2 ) {
       BX_FD_THIS s.status = 0x80;
       return;
     }
     BX_FD_THIS s.last_command.type            = (value & 0xf0) >> 4;
     BX_FD_THIS s.last_command.steprate        = (value & 0x03);
     BX_FD_THIS s.last_command.verify          = (value & 0x04) != 0;
     BX_FD_THIS s.last_command.headload        = (value & 0x08) != 0;
     BX_FD_THIS s.last_command.update_trackreg = (value & 0x10) != 0;
     BX_FD_THIS s.last_command.multi_recored   = (value & 0x10) != 0;
     BX_FD_THIS s.last_command.address_mark    = (value & 0x01) != 0;
     BX_FD_THIS s.last_command.side_number     = (value & 0x08) != 0;
     BX_FD_THIS s.last_command.delay_15ms      = (value & 0x04) != 0;
     BX_FD_THIS s.last_command.side_compare    = (value & 0x02) != 0;
     BX_FD_THIS s.last_command.interrupt       = (value & 0x0f);
     switch(BX_FD_THIS s.last_command.type & 0xe) {
     case 0x2:
     case 0x4:
     case 0x6:
     case 0x8:
     case 0xa:
       BX_FD_THIS s.last_command.type &= 0xe;
     }
     switch(BX_FD_THIS s.last_command.type) {
     case 0x0:
       BX_FD_THIS s.track  = 0;
       BX_FD_THIS s.status = 0x04;
       return;

     case 0x8:
       if(BX_FD_THIS s.last_command.multi_recored) goto error;
       if(BX_FD_THIS s.last_command.delay_15ms) goto error;
       if(!BX_FD_THIS s.media_present[BX_FD_THIS s.selected_drive]) {
	 BX_FD_THIS s.status = 0x10;
	 return;
       }
       if(BX_FD_THIS s.sector <  1 ||
	  BX_FD_THIS s.sector >
	  BX_FD_THIS s.media[BX_FD_THIS s.selected_drive].sectors_per_track ||
	  //BX_FD_THIS s.track  <  0 ||
	  BX_FD_THIS s.track  >=
	  BX_FD_THIS s.media[BX_FD_THIS s.selected_drive].tracks) {
	 BX_FD_THIS s.status = 0x10;
	 return;
       }
       goto error;
       return;
       
     default:
     error:
       BX_PANIC(("floppy: unknown command %02x\n", (unsigned) value));
     }

   case 0x0202:
     BX_FD_THIS s.track = value;
     return;

   case 0x0204:
     BX_FD_THIS s.sector = value;
     return;

   case 0x0206:
     BX_FD_THIS s.data=value;
     return;

   case 0x0208:
     BX_FD_THIS s.CLKSEL = (value & 0x20) != 0;
     BX_FD_THIS s.MOTOR  = (value & 0x10) != 0;
     BX_FD_THIS s.HDISEL = (value & 0x04) != 0;
     BX_FD_THIS s.DDEN   = (value & 0x02) != 0;
     BX_FD_THIS s.IRQMSK = (value & 0x01) != 0;
     return;

   case 0x020c:
     switch(value&0x0f) {
     case 0x00:
       BX_FD_THIS s.selected_drive=-1;
       return;
     case 0x01:
       BX_FD_THIS s.selected_drive=0;
       break;
     case 0x02:
       BX_FD_THIS s.selected_drive=1;
       break;
     case 0x04:
       BX_FD_THIS s.selected_drive=2;
       break;
     case 0x08:
       BX_FD_THIS s.selected_drive=3;
       break;
     }
     BX_FD_THIS s.INUSE[BX_FD_THIS s.selected_drive] = (value & 0x10) != 0;
     BX_FD_THIS s.HISPD[BX_FD_THIS s.selected_drive] = (value & 0x40) != 0;
     return;

   default:
      BX_PANIC(("floppy: io_write: unknown port %04x\n", (unsigned) address));
      break;
    }
}



  void
bx_floppy_ctrl_c::floppy_xfer(Bit8u drive, Bit32u offset, Bit8u *buffer,
            Bit32u bytes, Bit8u direction)
{
  int ret;

  if (drive > 1)
    BX_PANIC(("floppy_xfer: drive > 1\n"));

  if (bx_dbg.floppy) {
    BX_INFO(("drive=%u\n", (unsigned) drive));
    BX_INFO(("offset=%u\n", (unsigned) offset));
    BX_INFO(("bytes=%u\n", (unsigned) bytes));
    BX_INFO(("direction=%s\n", (direction==FROM_FLOPPY)? "from" : "to"));
    }

#ifdef macintosh
  if (strcmp(bx_options.floppya.path, SuperDrive))
#endif
    {
    ret = lseek(BX_FD_THIS s.media[drive].fd, offset, SEEK_SET);
    if (ret < 0) {
      BX_PANIC(("could not perform lseek() on floppy image file\n"));
      }
    }

  if (direction == FROM_FLOPPY) {
#ifdef macintosh
    if (!strcmp(bx_options.floppya.path, SuperDrive))
      ret = fd_read((char *) buffer, offset, bytes);
    else
#endif
      ret = ::read(BX_FD_THIS s.media[drive].fd, (bx_ptr_t) buffer, bytes);
    if (ret < int(bytes)) {
      /* ??? */
      if (ret > 0) {
        BX_INFO(("partial read() on floppy image returns %u/%u\n",
          (unsigned) ret, (unsigned) bytes));
        memset(buffer + ret, 0, bytes - ret);
        }
      else {
        BX_INFO(("read() on floppy image returns 0\n"));
        memset(buffer, 0, bytes);
        }
      }
    }

  else { // TO_FLOPPY
#ifdef macintosh
    if (!strcmp(bx_options.floppya.path, SuperDrive))
      ret = fd_write((char *) buffer, offset, bytes);
    else
#endif
      ret = ::write(BX_FD_THIS s.media[drive].fd, (bx_ptr_t) buffer, bytes);
    if (ret < int(bytes)) {
      BX_PANIC(("could not perform write() on floppy image file\n"));
      }
    }
}



  void
bx_floppy_ctrl_c::timer_handler(void *this_ptr)
{
#if defined(SIMX86)
  printf("Floppy timer\n");
#endif

  bx_floppy_ctrl_c *class_ptr = (bx_floppy_ctrl_c *) this_ptr;

  class_ptr->timer();
}

  void
bx_floppy_ctrl_c::timer()
{
#if 0
  switch ( BX_FD_THIS s.pending_command ) {
    default:
      BX_PANIC(("floppy:timer(): unknown case %02x\n",
        (unsigned) BX_FD_THIS s.pending_command));
    }
  return;
#endif
}

  void
bx_floppy_ctrl_c::dma_write(Bit8u *data_byte)
{
#if 0
  // A DMA write is from I/O to Memory
  // We need to return then next data byte from the floppy buffer
  // to be transfered via the DMA to memory. (read block from floppy)


  *data_byte = BX_FD_THIS s.floppy_buffer[BX_FD_THIS s.floppy_buffer_index++];

  if (BX_FD_THIS s.floppy_buffer_index >= 512) {
    Bit8u drive;

    drive = BX_FD_THIS s.DOR & 0x03;
    increment_sector(); // increment to next sector before retrieving next one
    BX_FD_THIS s.floppy_buffer_index = 0;
    if (bx_pc_system.TC) { // Terminal Count line, done
      BX_FD_THIS s.pending_command = 0;
      BX_FD_THIS s.main_status_reg = FD_MS_MRQ | FD_MS_DIO | FD_MS_BUSY;
      BX_FD_THIS s.result_size = 7;
      BX_FD_THIS s.result_index = 0;
      BX_FD_THIS s.result[0] = (BX_FD_THIS s.head[drive] << 2) | drive;
      BX_FD_THIS s.result[1] = 0;
      BX_FD_THIS s.result[2] = 0;
      BX_FD_THIS s.result[3] = BX_FD_THIS s.cylinder[drive];
      BX_FD_THIS s.result[4] = BX_FD_THIS s.head[drive];
      BX_FD_THIS s.result[5] = BX_FD_THIS s.sector[drive];
      BX_FD_THIS s.result[6] = 2;

      if (bx_dbg.floppy) {
        BX_INFO(("<<READ DONE>>\n"));
        BX_INFO(("AFTER\n"));
        BX_INFO(("  drive    = %u\n", (unsigned) drive));
        BX_INFO(("  head     = %u\n", (unsigned) BX_FD_THIS s.head[drive]));
        BX_INFO(("  cylinder = %u\n", (unsigned) BX_FD_THIS s.cylinder[drive]));
        BX_INFO(("  sector   = %u\n", (unsigned) BX_FD_THIS s.sector[drive]));
        }

      BX_FD_THIS devices->pic->trigger_irq(6);
      bx_pc_system.set_DRQ(FLOPPY_DMA_CHAN, 0);
      }
    else { // more data to transfer
      Bit32u logical_sector;
      logical_sector = (BX_FD_THIS s.cylinder[drive] * 2 *
                        BX_FD_THIS s.media[drive].sectors_per_track) +
                       (BX_FD_THIS s.head[drive] *
                        BX_FD_THIS s.media[drive].sectors_per_track) +
                       (BX_FD_THIS s.sector[drive] - 1);
      floppy_xfer(drive, logical_sector*512, BX_FD_THIS s.floppy_buffer,
                  512, FROM_FLOPPY);
      }
    }
#endif
}

  void
bx_floppy_ctrl_c::dma_read(Bit8u *data_byte)
{
#if 0
  // A DMA read is from Memory to I/O
  // We need to write the data_byte which was already transfered from memory
  // via DMA to I/O (write block to floppy)

  Bit8u drive;
  Bit32u logical_sector;

  BX_FD_THIS s.floppy_buffer[BX_FD_THIS s.floppy_buffer_index++] = *data_byte;

  if (BX_FD_THIS s.floppy_buffer_index >= 512) {
    drive = BX_FD_THIS s.DOR & 0x03;
    logical_sector = (BX_FD_THIS s.cylinder[drive] * 2 * BX_FD_THIS s.media[drive].sectors_per_track) +
                     (BX_FD_THIS s.head[drive] * BX_FD_THIS s.media[drive].sectors_per_track) +
                     (BX_FD_THIS s.sector[drive] - 1);
    floppy_xfer(drive, logical_sector*512, BX_FD_THIS s.floppy_buffer,
                512, TO_FLOPPY);
    increment_sector(); // increment to next sector after writing current one
    BX_FD_THIS s.floppy_buffer_index = 0;
    if (bx_pc_system.TC) { // Terminal Count line, done
      BX_FD_THIS s.pending_command = 0;
      BX_FD_THIS s.main_status_reg = FD_MS_MRQ | FD_MS_DIO | FD_MS_BUSY;
      BX_FD_THIS s.result_size = 7;
      BX_FD_THIS s.result_index = 0;
      BX_FD_THIS s.result[0] = (BX_FD_THIS s.head[drive] << 2) | drive;
      BX_FD_THIS s.result[1] = 0;
      BX_FD_THIS s.result[2] = 0;
      BX_FD_THIS s.result[3] = BX_FD_THIS s.cylinder[drive];
      BX_FD_THIS s.result[4] = BX_FD_THIS s.head[drive];
      BX_FD_THIS s.result[5] = BX_FD_THIS s.sector[drive];
      BX_FD_THIS s.result[6] = 2;
      if (bx_dbg.floppy) {
        BX_INFO(("<<WRITE DONE>>\n"));
        BX_INFO(("AFTER\n"));
        BX_INFO(("  drive    = %u\n", (unsigned) drive));
        BX_INFO(("  head     = %u\n", (unsigned) BX_FD_THIS s.head[drive]));
        BX_INFO(("  cylinder = %u\n", (unsigned) BX_FD_THIS s.cylinder[drive]));
        BX_INFO(("  sector   = %u\n", (unsigned) BX_FD_THIS s.sector[drive]));
        }

      BX_FD_THIS devices->pic->trigger_irq(6);
      bx_pc_system.set_DRQ(FLOPPY_DMA_CHAN, 0);
      }
    else { // more data to transfer
      } // else
    } // if BX_FD_THIS s.floppy_buffer_index >= 512
#endif
}



  void
bx_floppy_ctrl_c::increment_sector(void)
{
#if 0
  Bit8u drive;

  drive = BX_FD_THIS s.DOR & 0x03;

  // values after completion of data xfer
  // ??? calculation depends on base_count being multiple of 512
  BX_FD_THIS s.sector[drive] ++;
  if (BX_FD_THIS s.sector[drive] > BX_FD_THIS s.media[drive].sectors_per_track) {
    BX_FD_THIS s.sector[drive] -= BX_FD_THIS s.media[drive].sectors_per_track;
    BX_FD_THIS s.head[drive] ++;
    if (BX_FD_THIS s.head[drive] > 1) {
      BX_FD_THIS s.head[drive] = 0;
      BX_FD_THIS s.cylinder[drive] ++;
      if (BX_FD_THIS s.cylinder[drive] >= BX_FD_THIS s.media[drive].tracks) {
        // Set to 1 past last possible cylinder value.
        // I notice if I set it to tracks-1, prama linux won't boot.
        BX_FD_THIS s.cylinder[drive] = BX_FD_THIS s.media[drive].tracks;
        BX_INFO(("increment_sector: clamping cylinder to max\n"));
        }
      }
    }
#endif
}

  unsigned
bx_floppy_ctrl_c::set_media_status(unsigned drive, unsigned status)
{
  char *path;
  unsigned type;

  // if setting to the current value, nothing to do
  if (status == BX_FD_THIS s.media_present[drive])
    return(status);

  if (status == 0) {
    // eject floppy
    if (BX_FD_THIS s.media[drive].fd >= 0) {
      close( BX_FD_THIS s.media[drive].fd );
      BX_FD_THIS s.media[drive].fd = -1;
      }
    BX_FD_THIS s.media_present[drive] = 0;
    BX_FD_THIS s.DIR |= 0x80; // disk changed line
    return(0);
    }
  else {
    // insert floppy
    if (drive == 0) {
      path = bx_options.floppya.path;
      type = bx_options.floppya.type;
      }
    else {
      path = bx_options.floppyb.path;
      type = bx_options.floppyb.type;
      }
    if (evaluate_media(type, path, & BX_FD_THIS s.media[drive])) {
      BX_FD_THIS s.media_present[drive] = 1;
      BX_FD_THIS s.DIR |= 0x80; // disk changed line
      return(1);
      }
    else {
      BX_FD_THIS s.media_present[drive] = 0;
      return(0);
      }
    }
}

  unsigned
bx_floppy_ctrl_c::get_media_status(unsigned drive)
{
  return( BX_FD_THIS s.media_present[drive] );
}

  Boolean
bx_floppy_ctrl_c::evaluate_media(unsigned type, char *path, floppy_t *media)
{
  struct stat stat_buf;
  int ret;
  char sTemp[1024];

  if (type == BX_FLOPPY_NONE)
    return(0);

  // open media file (image file or device)
#ifdef macintosh
  media->fd = 0;
  if (strcmp(bx_options.floppya.path, SuperDrive))
#endif
    media->fd = open(path, O_RDWR
#ifdef O_BINARY
                   | O_BINARY
#endif
                  );

  if (media->fd < 0) {
    BX_INFO(( "floppy open of %s:\n",path,strerror(errno) ));
    return(0);
    }

#if BX_WITH_MACOS
  if (!strcmp(bx_options.floppya.path, SuperDrive))
    ret = fd_stat(&stat_buf);
  else
    ret = fstat(media->fd, &stat_buf);
#elif BX_WITH_WIN32
  stat_buf.st_mode = S_IFCHR;
  // maybe replace with code that sets ret to -1 if the disk is not available
  ret = 0;
#else
  // unix
  ret = fstat(media->fd, &stat_buf);
#endif
  if (ret) {
    BX_PANIC(("fstat()'ing floppy 0 drive image file returns error!\n"));
    return(0);
    }

  if ( S_ISREG(stat_buf.st_mode) ) {
    // regular file
    switch (type) {
      case BX_FLOPPY_720K: // 720K 3.5"
        media->type              = BX_FLOPPY_720K;
	media->bytes_per_sector  = 512;
        media->sectors_per_track = 9;
        media->tracks            = 80;
        media->heads             = 2;
        break;
      case BX_FLOPPY_1_2: // 1.2M 5.25"
        media->type              = BX_FLOPPY_1_2;
	media->bytes_per_sector  = 512;
        media->sectors_per_track = 15;
        media->tracks            = 80;
        media->heads             = 2;
        break;
      case BX_FLOPPY_1_44: // 1.44M 3.5"
        media->type              = BX_FLOPPY_1_44;
        if (stat_buf.st_size <= 1474560) {
	  media->bytes_per_sector  = 512;
          media->sectors_per_track = 18;
          media->tracks            = 80;
          media->heads             = 2;
          }
        else if (stat_buf.st_size == 1720320) {
	  media->bytes_per_sector  = 512;
          media->sectors_per_track = 21;
          media->tracks            = 80;
          media->heads             = 2;
          }
	else if (stat_buf.st_size == 1763328) {
	  media->bytes_per_sector  = 512;
          media->sectors_per_track = 21;
          media->tracks            = 82;
          media->heads             = 2;
	  }
        else {
          BX_INFO(("evaluate_media: file '%s' of unknown size %lu\n",
            path, (unsigned long) stat_buf.st_size));
          return(0);
          }
        break;
      case BX_FLOPPY_2_88: // 2.88M 3.5"
        media->type              = BX_FLOPPY_2_88;
	media->bytes_per_sector  = 512;
        media->sectors_per_track = 36;
        media->tracks            = 80;
        media->heads             = 2;
        break;
      case BX_FLOPPY_1_2J: // 1.2M(PC98)
        media->type              = BX_FLOPPY_1_2J;
	media->bytes_per_sector  = 1024;
        media->sectors_per_track = 8;
        media->tracks            = 77;
        media->heads             = 2;
        break;
      default:
        BX_PANIC(("floppy: evaluate_media: unknown media type\n"));
      }
    media->sectors = media->heads * media->tracks * media->sectors_per_track;
    return(1); // success
    }

  else if ( S_ISCHR(stat_buf.st_mode)
#if BX_WITH_MACOS == 0
#ifdef S_ISBLK
            || S_ISBLK(stat_buf.st_mode)
#endif
#endif
           ) {
    // character or block device
    // assume media is formatted to typical geometry for drive
    switch (type) {
      case BX_FLOPPY_720K: // 720K 3.5"
        media->type              = BX_FLOPPY_720K;
	media->bytes_per_sector  = 512;
        media->sectors_per_track = 9;
        media->tracks            = 80;
        media->heads             = 2;
        break;
      case BX_FLOPPY_1_2: // 1.2M 5.25"
        media->type              = BX_FLOPPY_1_2;
	media->bytes_per_sector  = 512;
        media->sectors_per_track = 15;
        media->tracks            = 80;
        media->heads             = 2;
        break;
      case BX_FLOPPY_1_44: // 1.44M 3.5"
        media->type              = BX_FLOPPY_1_44;
	media->bytes_per_sector  = 512;
        media->sectors_per_track = 18;
        media->tracks            = 80;
        media->heads             = 2;
        break;
      case BX_FLOPPY_2_88: // 2.88M 3.5"
        media->type              = BX_FLOPPY_2_88;
	media->bytes_per_sector  = 512;
        media->sectors_per_track = 36;
        media->tracks            = 80;
        media->heads             = 2;
        break;
      case BX_FLOPPY_1_2J: // 1.2M(PC98)
        media->type              = BX_FLOPPY_1_2J;
	media->bytes_per_sector  = 1024;
        media->sectors_per_track = 8;
        media->tracks            = 77;
        media->heads             = 2;
        break;
      default:
        BX_PANIC(("evaluate_media: unknown media type\n"));
      }
    media->sectors = media->heads * media->tracks * media->sectors_per_track;
    return(1); // success
    }
  else {
    // unknown file type
    fprintf(stderr, "# floppy: unknown mode type\n");
    BX_INFO(("floppy: unknown mode type\n"));
    return(0);
    }
}
#endif
